#include "motors.h"
#include <Arduino.h>

#ifndef cbi
#define cbi(sfr, bit) (_SFR_BYTE(sfr) &= ~_BV(bit))
#endif
#ifndef sbi
#define sbi(sfr, bit) (_SFR_BYTE(sfr) |= _BV(bit))
#endif 

// volatile Variablen zur Auswertung der Lochräder
static volatile long lastTicks[2]={0,0};
static volatile long numTicks[2]={0,0};
static volatile long dt[2]={0,0};

// Routine, von Interrupt-Routinen aufgerufen
static void tick(int i) {
  long us = micros();
  dt[i] = us-lastTicks[i];
  lastTicks[i]=us;
  numTicks[i]++;
}

// Interrupt-Routinen
static void leftTick() { tick(0); }
static void rightTick() { tick(1); }

void Motors::resetCounters() {
  for (int i=0; i<2; i++) {
    numTicks[i] = 0;
    // dt wird auch nicht resetted.
    // lastTicks wird nicht resetted, damit dt beim nächsten Tick bereits stimmt.
  }
}
long Motors::getTicks(int lr) {
  if (lr<0) lr = 0;
  if (lr>1) lr = 1;
  return numTicks[lr];
}
long Motors::getDt(int lr) {
  if (lr<0) lr = 0;
  if (lr>1) lr = 1;
  return dt[lr];
}
long Motors::getLastTicks(int lr) {
  if (lr<0) lr = 0;
  if (lr>1) lr = 1;
  return lastTicks[lr];
}
long* Motors::getTicksArray() {
  return numTicks;
}
long* Motors::getDtArray() {
  return dt;
}
long* Motors::getLastTicksArray() {
  return lastTicks;
}


static void setPWM(int pin, int power) {
  if (power>=PWM_MAX) power=PWM_MAX-1;
  switch(pin) {
    case 5:
      OCR3A = power;
      sbi(TCCR3A, COM3A1);
      break;
    case 6:
      sbi(TCCR4A, COM4A1);
      #if defined(COM4A0)             // only used on 32U4
        cbi(TCCR4A, COM4A0);
      #endif
      OCR4A = power;
      break;
    case 7:
      sbi(TCCR4A, COM4B1);
      OCR4B = power;
      break;
    case 8:
      sbi(TCCR4A, COM4C1);
      OCR4C = power;
      break;
  }
}


void Motors::begin() {

  // Set pins
  for (int i=0; i<4; i++) {
    pinMode(mpins[i],OUTPUT);
    digitalWrite(mpins[i],LOW);
    if (i<2) {
      pinMode(spins[i], INPUT);
      digitalWrite(spins[i], LOW);
    }
  }

    // 20 kHz -> Werte von 0 bis 799 (-> 16MHz Systemtakt)
  // timer 3: pins 2,3,5
  // timer 4: pins 6,7,8  
  TCCR3A = _BV(COM1A1) | _BV(COM1B1)  /* non-inverting PWM */
      | _BV(WGM11);                   /* mode 14: fast PWM, TOP=ICR3 */
  TCCR3B = _BV(WGM13) | _BV(WGM12)
      | _BV(CS10);                    /* no prescaling */
  ICR3 = PWM_MAX;                      /* TOP counter value */

  TCCR4A = _BV(COM1A1) | _BV(COM1B1)  /* non-inverting PWM */
      | _BV(WGM11);                   /* mode 14: fast PWM, TOP=ICR4 */
  TCCR4B = _BV(WGM13) | _BV(WGM12)
      | _BV(CS10);                    /* no prescaling */
  ICR4 = PWM_MAX;                      /* TOP counter value */

  // Encoders on pins 2 & 3
  attachInterrupt(digitalPinToInterrupt(2), leftTick, FALLING);
  attachInterrupt(digitalPinToInterrupt(3), rightTick, FALLING);

}

void Motors::setPower(int wheel, int power) {
  int dir = (power>=0) ? 0 : 1;
  setPWM(mpins[wheel*2+1-dir],0);
  digitalWrite(mpins[wheel*2+1-dir],0);
  setPWM(mpins[wheel*2+dir],abs(power));
}

void Motors::setPowers(int leftPower, int rightPower) {
  setPower(0, leftPower);
  setPower(1, rightPower);
}

