#ifndef MOTORS_INCLUDED
#define MOTORS_INCLUDED

// 800 for 20kHz = 16MHz/800 PWM
#define PWM_MAX 800

/**
 * Motorensteuerung.
 */

class Motors {

 public:
  /**
   * Initialisierung (wird von der Robot-Klasse aufgerufen).
   */
  void begin();
  /**
   * Legt die Leistung eines Motors fest.
   * @param lr links 0 oder rechts 1.
   * @param power Wert zwischen -799 und +799
   */
  void setPower(int lr, int power);

  /**
   * Legt die Leistung beider Motoren fest.
   * @param leftPower Wert zwischen -799 und +799
   * @param rightPower Wert zwischen -799 und +799
   */
  void setPowers(int leftPower, int rightPower);


  /** Liefert die Anzahl Ticks eines Motors 0=links, 1=rechts zurück.
   *  @param lr 0 (links) oder 1 (rechts)
   *  @return Anzahl Ticks
   */
  long getTicks(int lr);

  /**
   * Liefert die Anzahl Microsekunden eines Motors (0=links, 1=rechts) zwischen den beiden letzten Ticks zurück.
   * Achtung: Das Resultat macht erst dann Sinn, wenn die Anzahl Ticks>=2.
   * @param lr 0 (links) oder 1 (rechts)
   * @return Anzahl Microsekunden.
   */
  long getDt(int lr);
  /**
   * Liefert den Zeitpunkt des letzten Ticks als Systemzeit in Microsekunden eines Motors (0=links, 1=rechts) zwischen den beiden letzten Ticks zurück.
   * @param lr 0 (links) oder 1 (rechts)
   * @return Zeitpunkt in Microsekunden Systemzeit.
   */
  long getLastTicks(int lr);

  /**
   * Liefert einen Pointer auf ein long-Array mit 2 Einträgen (links / rechts) zurück. Dieses Array wird überschrieben.
   * Damit können die Anzahl Ticks ausgelesen werden, ohne die Funktion \code getTicks(int) \endcode aufzurufen.
   * 
   * Beispiel:
   * \code
   *   long* ticks = robot.motors.getTicksArray(); 
   *   // Fortan steht in ticks[0] und ticks[1] die Anzahl Ticks vom linken und rechten Motor.
   *   if (ticks[0]>10) { // Wenn linker Motor mehr als halbe Umdrehung
   *      // Tu was
   *   }
   * \endcode
   * @return Pointer auf long-Array mit 2 Einträgen.
   */
  long* getTicksArray();

  /**
   * Liefert einen Pointer auf ein long-Array mit 2 Einträgen (links / rechts) zurück. Dieses Array wird überschrieben.
   * Damit kann die Zeit zwischen zwei Ticks ausgelesen werden, ohne die Funktion \code getDt(int) \endcode aufzurufen.
   * Siehe auch getTicksArray().
   * @return Pointer auf long-Array mit 2 Einträgen.
   */  
  long* getDtArray();

  
  /**
   * Liefert einen Pointer auf ein long-Array mit 2 Einträgen (links / rechts) zurück. Dieses Array wird überschrieben.
   * Damit kann die Systemzeit der letzten Ticks ausgelesen werden, ohne die Funktion \code getLastTicks(int) \endcode aufzurufen.
   * Siehe auch getTicksArray().
   * @return Pointer auf long-Array mit 2 Einträgen.
   */  
  long* getLastTicksArray();

  /**
   * Setzt die Ticks Zähler auf Null. Eine direkte Manipulation ist auch über die Arrays möglich.
   */
  void resetCounters();

  
  private:
    // Left fwd, Left bkwd, Right fwd, Right bkwd
    int mpins[4] = {5,6,7,8};  // PWM-pins on timers 3 (2,3,5) and 4 (6,7,8)
    // sensor pins: left, right
    int spins[2]={2,3};
};

#endif
