
class Zelle:
    """Ein Quadrätchen eines Labyrinths

    Darin ist gespeichert, ob eine Wand offen oder zu ist, eine
    Markierung und eine Referenz auf das Labyrinth selbst.

    Die Richtungen sind wie folgt (im trigonometrischen Umlaufsinn, 
    d.h. von der x- zur y-Achse, nur dass die y-Achse hier nach unten zeigt.)

        3
        ^
    2 <-o-> 0   positive x-Achse
        v
        1
    positive y-Achse

    Attributes
    ----------
    x : int
        x-Koordinate der Zelle
    y : int
        y-Koordinate der Zelle
    mark : str
        Markierung (Zeichenkette der Länge 1)
    offen : list[bool]
        True (offen) oder False (zu) für jede Wand in der
        entsprechenden Richtung
    bild : int
        0 (schwarz) oder 1 (weiss)
        wird gesetzt, wenn im Labyrinth ein Bild geladen wird.

    Methods
    -------
    zustand(dir:int, offenZu:bool)
        Öffnet (offenZu=True) oder schliesst (offenZu=False) die
        Wand in Richtung dir (Automatische Anpassung in der Nachbarszelle)

    nachbar(dir:int)
        Liefert die Nachbarszelle in Richtung dir, oder None, wenn
        es diese nicht gibt.

    dirTo(other:Zelle)
        Liefert die Richtung, in der sich die Nachbarzelle other befindet.
        (Wird z.B bei der Wegfindung gebraucht.)
    """
    def __init__(self, x:int, y:int, laby):
        """
        Parameters
        ----------
        x : int
            x-Koordinate der Zelle
        y : int
            y-Koordinate der Zelle
        laby : Laby
            Laby-Objekt
        """
        self.x = x
        self.y = y
        self.laby = laby
        # Erst mal leere Markierung (Leerschlag)
        # Die Markierung muss eine Zeichenkette der Länge 1 sein.
        self.mark = " "
        # Bildpixel für die diese Zelle ist erst mal schwarz
        self.bild = 0
        # Erst mal alle Wände zu
        self.offen = [False, False, False, False]

    def zustand(self, dir:int, offenZu:bool) -> None:
        """
        Öffnet (True) oder schliesst (False) Wände in gegebener Richtung

        Parameters
        ----------
        dir : int
            Richtung (0-3), in der die Wand geöffnet oder geschlossen werden soll.
        offenZu : bool
            True (öffnen) oder False (schliessen)
        """
        self.offen[dir] = offenZu            # Info auf der aktuellen Zelle setzen
        nb = self.nachbar(dir)               # Nachbar in dieser Richtung finden
        if nb!=None:
            nb.offen[(dir+2)%4] = offenZu    # Info auf dem Nachber setzen (Richtung um 180 Grad gedreht)

    def nachbar(self, dir:int):
        """
        Liefert die Nachbarszelle in Richtung dir (oder None, wenn es diese nicht gibt)
        Parameters
        ----------
        dir : int
            Richtung (0-3), in welcher die Nachbarszelle bestimmt werden soll.

        Returns
        -------
        Nachbarszelle oder None
        """
        # Umwandlung der Richtung in eine Liste mit 2 Einträgen
        offset = [[1,0],[0,1],[-1,0],[0,-1]][dir]
        nbx = self.x+offset[0]
        nby = self.y+offset[1]
        if self.laby.on(nbx, nby):
            return self.laby[nbx, nby]
        return None   # Diesen Nachbar gibt es gar nicht

    # Richtung von dieser Zelle zu einer Nachbarzelle (other)
    def dirTo(self, other) -> int:
        """Liefert die Richtung, in der sich eine Nachbarszelle befindet

        Parameters
        ----------
        other : Zelle
            Nachbarszelle, zu der die Richtung bestimmt werden soll
        
        Returns
        -------
        Richtung als Zahl 0-3.
        """
        if other.x>self.x:
            return 0
        if other.x<self.x:
            return 2
        if other.y>self.y:
            return 1
        return 3

    # Infos zu dieser Zelle
    def __str__(self) -> str:
        return f"({self.x},{self.y}) offen:[{self.offen[0]},{self.offen[1]},{self.offen[2]},{self.offen[3]}]"

