/*
Installing the PubSubClient Library
The PubSubClient library provides a client for doing simple publish/subscribe messaging
with a server that supports MQTT (basically allows your ESP32 to talk with Node-RED).

Click here ( https://github.com/knolleary/pubsubclient/archive/master.zip )
to download the PubSubClient library. You should have a .zip folder in your Downloads folder
Unzip the .zip folder and you should get pubsubclient-master folder
Rename your folder from pubsubclient-master to pubsubclient
Move the pubsubclient folder to your Arduino IDE installation libraries folder
Then, re-open your Arduino IDE.

from https://randomnerdtutorials.com/esp32-mqtt-publish-subscribe-arduino-ide/
 */

#include <WiFi.h>
#include <PubSubClient.h>  // API: See https://pubsubclient.knolleary.net/api

#include "secrets.h"   // Passwords, Server etc.
#define BUILTIN 2   // Onboard LED

WiFiClient espClient;              // WiFi-Verbindung
PubSubClient client(espClient);    // MQTT

void setup_wifi() {
  Serial.printf("\nConnecting to %s\n",SECRET_WIFI_SSID);
  WiFi.begin(SECRET_WIFI_SSID, SECRET_WIFI_PASSWD);
  while (WiFi.status() != WL_CONNECTED) {
    Serial.print(".");
    delay(500);
  }
  Serial.printf("\nWiFi connected\nIP address: %s\n",WiFi.localIP());
}


/*
 * Wird aufgerufen, wenn eine neue Nachricht zu einem abonnierten Topic reinkommt
 */
void callback(char* topic, byte* message, unsigned int length) {
  String messageTemp;
  for (int i = 0; i < length; i++) {
    Serial.print((char)message[i]);
    messageTemp += (char)message[i];
  }
  Serial.printf("Message arrived on topic: %s. Message %s\n", topic, messageTemp);
  if (String(topic) == "ffprg20/ivo/led") {  // Led ein- ausschalten.
    digitalWrite(BUILTIN, messageTemp == "1");
  }
}


/*
 * Stellt die Verbindung mit dem MQTT-Server her.
 */
void reconnect() {
  // Loop until we're reconnected
  while (!client.connected()) {
    // make unique client ID
    uint64_t chipid=ESP.getEfuseMac();
    char clientID[20];
    sprintf(clientID, "esp32-%x", (uint32_t)(chipid & 0xffffffff));
    
    Serial.print("Attempting MQTT connection...");
    if (client.connect(clientID, SECRET_MQTT_USERNAME, SECRET_MQTT_PASSWD)) {
      Serial.printf("connected with id %s\n", clientID);
      // Subscribe to topic(s)
      client.subscribe("ffprg20/ivo/led");
    } else {
      Serial.printf("failed, state=%d, (see https://pubsubclient.knolleary.net/api#state)\n", client.state());
    }
  }
}


/*
 * Misst den internen Hallsensor und gibt die gemittelte Messung
 * jede Sekunde per mqtt aus.
 */
void showHall() {
  static unsigned long lastMsg = 0;
  static long sum = 0;
  static unsigned long num = 0;
  sum += hallRead();
  num++;
  if (millis()-lastMsg > 1000) { // Jede Sekunde Messwert absetzen
    String magnet = String((float)sum/num); 
    bool ok = client.publish("ffprg20/ivo/hall", magnet.c_str());
    Serial.printf("Hall Sensor meldet %s, published %d, #Messungen=%d\n", magnet.c_str(),ok, num);
    lastMsg = millis();
    sum = 0;
    num = 0;
  }
}

void setup() {
  Serial.begin(115200);
  pinMode(BUILTIN, OUTPUT);
  setup_wifi();
  client.setServer(SECRET_MQTT_SERVER, SECRET_MQTT_PORT);
  client.setCallback(callback);
  
}

void loop() {
  client.loop();
  showHall();
  if (!client.connected()) {
    reconnect();
  } 
}
