
import board
Board = board.Board

import params
Params = params.Params

import strategy
Strategy = strategy.Strategy

import ivo_simple_strategy 
IvoSimpleStrategy = ivo_simple_strategy.IvoSimpleStrategy


import time
import random
import sys
import traceback


class Game:
    """Verwaltet die Strategieen und koordiniert das Spiel
    
    Attributes:
        strategies: Array mit Strategy-Instanzen
        params: Array mit Params-Instanzen (1 pro Spieler)
        positions: Array mit Positionen der Schlangenkoepfe
        feld: Aktuelles Feld
        time (int): Zeit (in Spielschritten)
        vanish: Array mit Zeitpunkten, wann der Feldeintrag geloescht werden soll.
        lengths: Laenge der Schlangen
        changed: (int[][2]) Array mit Positionen, die sich geaendert haben.
        appletime: (int): Zeitpunkt, zu dem alle Schlangen wegen verfaultem Apfel sterben.
    """
    
    APPLEPOINTS=100
    """Punkte pro Apfel"""
    
    APPLELENGTH=10
    """Laengenzuwachs pro Apfel"""
    
    APPLETIME=400
    """Zeit, bis der Apfel verfault und in der Folge alle Schlangen sterben"""
        

    def __init__(self, board, strategyClasses, graphics=False):
        """Args:
            board (Board): Spielfeld, auf dem gespielt werden soll
            strategyClass (Array of Strategy): 1-4 Strategien, die gegeneinander antreten sollen
            graphics (bool): Ob ein Grafikfenster geoeffnet werden soll.
        """
        self.graphics = graphics
        self.strategies=list(map(lambda c:c(board.width, board.height), strategyClasses))
        self.params=list(map(lambda s:s.getParams(), self.strategies))
        self.positions = list(map(lambda x:x[0:2], board.startPos))
        self.oldpositions = list(map(lambda x:x[0:2], board.startPos))
        self.dirs = list(map(lambda x:x[2], board.startPos))
        self.olddirs = list(map(lambda x:x[2], board.startPos))
        self.width = board.width
        self.height = board.height
        self.feld = [[board.feld[j][i] for i in range(0,self.height)] for j in range(0,self.width)]
        self.vanish = [[0 for i in range(0,self.height)] for j in range(0,self.width)]
        self.time = 1
        self.n = len(strategyClasses)
        self.alive = [True for i in range(0,self.n)]
        self.lengths = [5 for i in range(0,self.n)]
        self.points = [0 for i in range(0,4)]
        self.appletime = self.APPLETIME
        self.changed=[]
        self.timeConsumed = [0.0 for i in range(self.n)]
        

        if self.graphics:
            from snakegraphics import SnakeGraphics
            self.graphics = SnakeGraphics(self.feld)

        for i in range(0,self.n):
            p = self.positions[i]
            self.updateBoard(p[0],p[1],i+1+self.dirs[i]*10+10, self.time+self.lengths[i])

            
        self.placeApple()
        self.copyParams(False)
        
    def __str__(self):
        return Board.str(self.feld);
        
    def copyParams(self, quick=True):
        """Kopiert alle Parameter in die noch lebenden Strategien"""
        heads = []
        for i in range(self.n):
            s=self.strategies[i]
            if self.alive[i]:
                heads.append(self.positions[i])
                p=s.getParams()
                if not quick:
                    for x in range(0,self.width):
                        for y in range(0,self.height):
                            p.feld[x][y] = self.feld[x][y]
                            p.vanish[x][y] = self.vanish[x][y]
                p.alive = self.alive[i]
                p.time = self.time
                p.dir = self.dirs[i]
                p.x = self.positions[i][0]
                p.y = self.positions[i][1]
                p.appletime = self.appletime
                p.changed = [x for x in self.changed]
                p.apple = [x for x in self.apple]
        for i in range(self.n):
            if self.alive[i]:
                self.strategies[i].getParams().heads = [h for h in heads]
    
    def updateBoard(self, x,y,what, vanish):
        """Update des felds x,y auf what. Effiziente Grafik und effizientes update der Strategien"""
        # print("   -> feld[%d][%d]=%d" %(x,y,what))
        self.feld[x][y] = what
        if (vanish>=0):
            self.vanish[x][y] = vanish
        self.changed.append([x,y])
        if self.graphics:
            self.graphics.draw(x,y,what)
        for i in range(0,self.n):
            if self.alive[i]:
                self.strategies[i].getParams().feld[x][y] = what
                if (vanish>=0):
                    self.strategies[i].getParams().vanish[x][y] = vanish
                
    
    def emptyPlace(self):
        """Liefert ein zufaelliges leeres Feld (fuer die Platzierung des Apfels)"""
        while True:
            p = [random.randint(1,self.width-2), random.randint(1,self.height-2)]
            if self.feld[p[0]][p[1]]==Board.EMPTY:
                return p
            
                
    def minDist(self,p):
        """Liefert die kleinste Distanz (Manhatten-Metrik) zu einem Schlangenkopf. 
        Damit wird der Apfel moeglichst nicht direkt vor einer Schlange platziert."""
        m=self.width+self.height+2;
        for i in range(0,self.n):
            if self.alive[i]:
                d = abs(p[0]-self.positions[i][0])+abs(p[1]-self.positions[i][1])
                if d<m:
                    m = d
        return m
    
            
    def placeApple(self):
        """Platziert den Apfel 20 mal zufaellig und nimmt die Position mit 
        groesster minimaler Distanz zum naechsten Schlangekopf"""
        p = self.emptyPlace()
        d = self.minDist(p)
        for i in range(20):
            pp = self.emptyPlace()
            dd = self.minDist(pp)
            if (dd>d):
                p = pp
                d = dd
        self.apple=p
        self.appletime = self.time+self.APPLETIME
        self.updateBoard(p[0],p[1],Board.APPLE, self.appletime)
            
        

    def step(self):
        """Fuehrt eine Spielzug aus."""
        self.changed = []
        # Remove tails (still very inefficient)
        for x in range(1,self.width-1):
            for y in range(1,self.height-1):
                if self.vanish[x][y]==self.time:
                    self.updateBoard(x,y,Board.EMPTY,0)
                    
        # Get directions
        for i in range(0,self.n):
            if self.alive[i]:
                try: # Avoid runtime exceptions
                    self.dirs[i] = self.strategies[i].getDir()
                except Exception as e: # Report them properly, if they happen
                    self.alive[i] = False
                    info = sys.exc_info()
                    self.params[i].reason = "Fehler im Programm: "+str(info)+"\n"+"".join(traceback.format_tb(info[2]))
                    print(self.params[i].reason)
        # Advance Positions
        for i in range(0,self.n):
            if self.alive[i]:
                self.positions[i][0]+=Board.VECS[self.dirs[i]][0]
                self.positions[i][1]+=Board.VECS[self.dirs[i]][1]
        for i in range(0,self.n):
            if self.alive[i]:
                # Check for head-to-head collisions
                # TODO: Betreten 3 Schlangen das gleiche Feld, ueberlebt die Schlange mit
                #       hoechstem index...
                for j in range(i+1,self.n):
                    if self.alive[j]:
                        if self.positions[i]==self.positions[j]:
                            self.alive[i]=False
                            self.alive[j]=False
                            self.params[i].reason = "Kopf an Kopf Zusammenstoss"
                            self.params[j].reason = "Kopf an Kopf Zusammenstoss"
            if self.alive[i]:
                # Check for other collisions and apple eating
                p = self.positions[i]
                if self.feld[p[0]][p[1]]==Board.APPLE:
                    self.placeApple()
                    self.points[i]+=self.APPLEPOINTS
                    self.lengths[i]+=self.APPLELENGTH
                elif self.feld[p[0]][p[1]]!=Board.EMPTY:
                    self.alive[i]=False
                    self.params[i].reason = "In eine Wand/Schlange gedonnert"
                    
        for i in range(0,self.n):
            if self.alive[i]:
                self.updateBoard(self.positions[i][0], self.positions[i][1], i+1+(self.dirs[i]+1)*10, self.time+self.lengths[i]);
                self.updateBoard(self.oldpositions[i][0], self.oldpositions[i][1], i+1+Board.bodyEntry(self.olddirs[i],self.dirs[i]), -1);
                for j in range(0,2):
                    self.oldpositions[i][j]=self.positions[i][j]
                self.olddirs[i] = self.dirs[i]
                self.vanish[self.positions[i][0]][self.positions[i][1]] = self.time+self.lengths[i]
                self.points[i]+=1
                
        # Verfaulter Apfel?
        if (self.time==self.appletime):
            for i in range(self.n):
                if self.alive[i]:
                    self.alive[i] = False
                    self.params[i].reason = "Vom verfaulten Apfel vergiftet"
        
        self.time+=1                                                                                            
        self.copyParams()

        

    def run(self, delaySeconds=0):
        """Laesst das Spiel laufen
        
        Args:
            delaySeconds (float): Anzahl Sekunden, die zwischen zwei Spielschritten gewartet wird.
            
        Returns:
            Array mit erreichten Punkten
        """
        while True:
            self.step()
            if delaySeconds>0:
                time.sleep(delaySeconds)
            if not self.graphics:
                print(Board.str(self.feld))
            if not any(self.alive):
                break;
        if self.graphics:
            time.sleep(1)
            self.graphics.dispose()
        return self.points
    
    

if __name__=="__main__":
    strats = [IvoSimpleStrategy, IvoSimpleStrategy, IvoSimpleStrategy, IvoSimpleStrategy]
    board = Board(30,20)
    game = Game(board, strats, True)
    print(Board.str(game.feld))
    res = game.run(0.5)
    print(res)
    print([game.params[i].reason for i in range(game.n)])
    
    
