/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package snake.server;

import java.util.Random;
import snake.GameConstants;

/**
 *
 * @author ivo
 */
public class SingleGameServer {
    private final Connection[] c;
    private final String[] errors;
    private final int[] states;
    private final int[] points;
    private final int[][] positions;
    private final int n;
    private int starving = 0;
    private final int[] lengths;
    private final Field field;
    private final Random rand;
    
    public SingleGameServer(Game game) {
        n = game.n;
        states = new int[n];
        positions = new int[n][];
        errors = new String[n];
        points = new int[n];
        lengths = new int[n];
        c = new Connection[n];
        this.field = game.field;
        for (int i=0; i<n; i++) {
            c[i] = game.connections[i];
            errors[i] = "";
            positions[i] = field.getPosition(i);
            lengths[i] = GameConstants.INI_LENGTH;
            field.setAge(positions[i], lengths[i]);
        }
        rand = new Random();
    }
    
    private void readMoves() {
        // Read MOVEs from alive players
        for (int p=0; p<n; p++) {
            if (states[p]>=0) {
                String line = c[p].nextLine();
                if (line==null) {
                    // This player is dead
                    states[p]=-1;
                    return;
                }
                switch(line.substring(line.indexOf(' ')+1)) {
                    case "right":
                        states[p]=0;
                        break;
                    case "down":
                        states[p]=1;
                        break;
                    case "left":
                        states[p]=2;
                        break;
                    case "up":
                        states[p]=3;
                        break;
                    default:
                        errors[p]="Expected MOVE up/down/left/right. Got ->"+line+"<- instead.";
                        states[p]=-2; // BAD MOVE
                }
            }
        }
    }
    
    private void processErrors() {
        for (int p=0; p<n; p++) {
            if (states[p]<-1) {
                c[p].println("DEAD\n"+points[p]+"\n"+errors[p]);
                states[p]=-1;
                c[p].close();
            }
        }
    }
    
    
    private void executeMoves() {
        int[][] vec = new int[][] {{1,0},{0,1},{-1,0},{0,-1}};
        
        // Positionen vorrücken, Kopfstösse aussortieren
        for (int p=0; p<n; p++) {
            if (states[p]>=0) {
                for (int i=0; i<2; i++) {
                    positions[p][i]+=vec[states[p]][i];
                }
                for (int pp=0; pp<p; pp++) {
                    if (states[pp]>=0 && positions[p][0] == positions[pp][0] && positions[p][1] == positions[pp][1]) {
                        states[pp] = -3;
                        errors[pp] = "Kopfweh! Zwei Schlangen auf einem Feld!";
                        states[p] = -3;
                        errors[p] = "Kopfweh! Zwei Schlangen auf einem Feld!";
                    }
                }
            }
        }
        // Schlange sterben lassen, Apfel fressen
        boolean newApple = false;
        for (int p=0; p<n; p++) {
            if (states[p]>=0) {
                char f = field.get(positions[p]);
                if (f=='.') {
                    field.set(positions[p], (char)('0'+p), lengths[p]);
                } else if (f=='A') {
                    points[p]+= GameConstants.APPLE_POINTS;
                    lengths[p]+=GameConstants.APPLE_LENGTH;
                    newApple = true;
                    starving = 0;
                } else if (f=='#') {
                    states[p]=-4;
                    errors[p]="In die Mauer gedonnert";
                } else if (f>='0' && f<='9') {
                    if (f-'0' != p) {
                        points[f-'0']+=GameConstants.CRASH_POINTS;
                        errors[p] = "Gegenerische Schlange gerammt!";
                    } else {
                        errors[p] = "Selbstmord!";
                    }
                    states[p]=-5;
                }
            }
        }
        // Überlebende vorrücken, Punkte erhöhen, oder Hungertod
        starving++;
        for (int p=0; p<n; p++) {
            if (states[p]>=0) {
                if (starving>=GameConstants.STARVING) {
                    states[p]=-6;
                    errors[p]="Apfel verschimmelt, Hungetod :-P";
                } else {
                    points[p]++;
                    field.set(positions[p], (char)('0'+p), lengths[p]);
                }
            }
        }
        if (newApple) {
            generateNewApple();
        }

    }
    
    private void submitChanges() {
        for (int p=0; p<n; p++) {
            if (states[p]>=0) {
                c[p].println("ALIVE");
                c[p].println(""+points[p]);
                c[p].println("UPDATE");
                field.writeHistory(c[p], p);
                if (!c[p].flush()) {
                    states[p] = -1;
                }
            }
        }
        field.clearHistory();
    }
    
    private boolean running() {
        int active = 0;
        for (int p=0; p<n; p++) {
            if (states[p]>=0) {
                if (c[p].alive()) {
                    active++;
                } else {
                    states[p]=-1;
                }
            }
        }
        return active>0;
    }
    
    private void generateNewApple() {
        while (true) {
            int x = rand.nextInt(field.width-1)+1;
            int y = rand.nextInt(field.height-1)+1;
            if (field.get(x,y)=='.') {
                int mind = field.width+field.height;
                for (int p=0; p<n; p++) {
                    int d = Math.abs(positions[p][0]-x)+Math.abs(positions[p][1]-y);
                    if (d<mind) mind=d;
                }
                if (mind>(field.height+field.width)/10) {
                    field.set(x,y, 'A', 0);
                    return;
                }
            }
        }
    }
    
    private void startGame() {
        generateNewApple();
        field.clearHistory();
        for (int p=0; p<n; p++) {
            c[p].println("START");
            c[p].println(String.format("%d %d %d", n, field.width, field.height));
            field.writeField(c[p], p);
            if (!c[p].flush()) {
                states[p] = -1;
            }
        }
    }
    
    public void run() {
        startGame();
        while (running()) {
            readMoves();
            processErrors();
            executeMoves();
            processErrors();
            field.doAgeStep();
            submitChanges();
        }
    }
    
}
