/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package snake.server;

import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.ServerSocket;
import java.net.Socket;
import java.util.HashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.logging.Level;
import java.util.logging.Logger;
import snake.GameConstants;

/**
 *
 * @author ivo
 */
public class Server implements Runnable {

    private static HashMap<String,Game> games = new HashMap<>();
    
    private Socket client;
            
    private Server(Socket client) {
        this.client = client;
    }
    
    private static void showGames(final Connection c) {
        if (!c.println("GAMES")) return;
        int count = (int) games.values().stream().filter(g -> g.pub).count();
        if (!c.println(""+count)) return;
        games.forEach((n,g) -> c.println(g.show()));
    }
    
    
    private static void newGame(final Connection c) {
        String pub = c.nextLine();
        if (pub==null) return;
        if (!(pub.equals("private") || pub.equals("public"))) {
            c.sendError("Second line must contain 'public' or 'private'");
            return;
        }
        String name = c.nextLine();
        if(name==null) return;
        if (name.length()<2) {
            c.sendError("Name ist too short!");
            return;
        }
        if (games.containsKey(name)) {
            c.sendError("There is already a game with the name ->"+name+"<-");
            return;
        }
        int numPlayers = c.nextInt();
        if (numPlayers>9 || numPlayers<1) {
            c.sendError("Too many (or to few) players, maximum 9, minimum 1!");
            return;
        }
        int w = c.nextInt();
        if (w>500 || w<10) {
            c.sendError("Maximum width is 500, minimum is 10. You've sent "+w);
            return;
        }
        int h = c.nextInt();
        if (h>500 || h<10) {
            c.sendError("Maximum height is 500, minimum is 10. You've sent "+h);
            return;
        }
        c.nextLine(); // Consume newline
        Field field = new Field(w,h,c);
        String error = field.checkField(numPlayers);
        if (error==null) {
            games.put(name, new Game(h, name, field, pub.equals("public")));
            c.println("OK");
        } else {
            c.sendError(error);
        }
    }
    
    private static Game joinGame(Connection c) {
        String name = c.nextLine();
        if (name==null) return null;
        if (games.containsKey(name)) {
            Game game = games.get(name);
            if (game.available() && game.join(c)) {
                c.println("OK");
                if (!c.println("Please wait")) return null;                
                if (!game.available()) {
                    Logger.getLogger(Server.class.getName()).log(Level.INFO, "Found game "+name);
                    return game;
                }
            } else {
                c.sendError("Das Spiel ist bereits voll.");
            }
        } else {
            c.sendError("Das Spiel existiert nicht oder nicht mehr.");
        }
        return null;
    }
    
    private static class GameSpec {
        public final String name;
        public final int w,h,n;
        public final String method;
        public GameSpec(String name, int w, int h, int n, String method) {
            this.name = name;
            this.w = w;
            this.h = h;
            this.n = n;
            this.method = method;
        }
        
        public Game create() {
            snake.fields.Field f = snake.fields.Field.create(w, h, n);
            if (method!=null) {
                try {
                    Method m = snake.fields.Field.class.getMethod(method);
                    m.invoke(f);
                } catch (NoSuchMethodException | SecurityException | IllegalAccessException | IllegalArgumentException | InvocationTargetException ex) {
                    Logger.getLogger(Server.class.getName()).log(Level.SEVERE, null, ex);
                }
            }
            return new Game(n, name, new Field(w,h,f.getLines()), true);
        }
        
    }
    
    private static final GameSpec[] GAMES = new GameSpec[] {
        new GameSpec("Plain single", 30,20,1,null),
        new GameSpec("Plain two", 60,40,2,null),
        new GameSpec("Plain four", 60,40,4,null),
        new GameSpec("Labywalls single", 60,40,1,"addLabyWalls"),
        new GameSpec("Labywalls two", 60,40,2,"addLabyWalls"),
        new GameSpec("Labywalls four", 60,40,4,"addLabyWalls"),
        new GameSpec("Three rooms single", 60,40,1,"addThreeRooms"),
        new GameSpec("Three rooms two", 60,40,2,"addThreeRooms"),
        new GameSpec("Three rooms four", 60,40,4,"addThreeRooms"),
        new GameSpec("Nine rooms single", 60,40,1,"addNineRooms"),
        new GameSpec("Nine rooms single", 60,40,2,"addNineRooms"),
        new GameSpec("Nine rooms single", 60,40,4,"addNineRooms"),
    };
    
    private static void makeDefaultGames() {
        for (GameSpec gs : GAMES) {
            if (!games.containsKey(gs.name)) {
                games.put(gs.name, gs.create());
            }
        }
    }

    @Override
    public void run() {
        try {
            Connection c = new Connection(client);
            //c.debug = true;
            Game game = null;
            while (game == null) {
                if (!c.alive()) {
                    break;
                }
                String line = c.nextLine();
                if (line==null) return;
                //Logger.getLogger(Server.class.getName()).log(Level.INFO, "Got "+line);
                switch (line) {
                case "GAMES?":
                    Server.showGames(c);
                    break;
                case "NEWGAME":
                    Server.newGame(c);
                    break;
                case "JOIN":
                    game = Server.joinGame(c);
                    break;
                }
            }
            if (game != null) {
                // Remove game and create replacement, if necessary
                games.remove(game.name);
                makeDefaultGames();
                new SingleGameServer(game).run();
            }
            if (!c.alive()) {
                c.close();
            }
        } catch (IOException ex) {
            Logger.getLogger(Server.class.getName()).log(Level.SEVERE, null, ex);
        }
    }
    
    
    public static void main(String[] args) {
        ExecutorService executor;
        try (ServerSocket socket = new ServerSocket(GameConstants.TCP_PORT)) {
            executor = Executors.newCachedThreadPool();
            Logger.getLogger(Server.class.getName()).log(Level.INFO, "Waiting for connections");
            makeDefaultGames();
            while (true) {
                Socket client = socket.accept();
                Runnable worker = new Server(client);
                executor.execute(worker);
            }
        } catch (IOException ex) {
            Logger.getLogger(Server.class.getName()).log(Level.SEVERE, null, ex);
            System.exit(-1);
        }
    }

}
