/*
 *  Copyright Ivo Blöchliger, ivo.bloechliger@unifr.ch
 *  University of Fribourg.
 *  You may use and modify this code for teaching and learning 
 *  purposes. For any other use, please contact the author.
 */

package ch.ksbg.fginfo;

import java.util.Stack;

/**
 *
 * @author ivo
 */
public class Turtle {
    
    private final Stack<State> stack = new Stack<>();
    private State state = new State();
    
    /**
     * Saves the current state of the turtle.
     * May be called multiple times, all states will be saved.
     */
    public void save() {
        stack.push(new State(state));
    }
    
    /**
     * Restores the last saved state of the turtle.
     * May be called as many times as save() was called.
     */
    public void restore() {
        if (!stack.empty()) {
            state = stack.pop();
        }
    }
    
    /**
     * Sets the x-coordinate of the turtle
     * @param x x-coordinate
     */
    public void setX(double x) {
        state.x = x;
    }
            
    /**
     * Sets the y-coordinate of the turtle
     * @param y y-coordinate
     */
    public void setY(double y) {
        state.y = y;
    }
    
    /** 
     * Sets the angle of the turtle in degrees. 
     * @param a angle in degrees
     */
    public void setAngle(double a) {
        state.a = a/180.0*Math.PI;
    }
    
    /**
     * Sets the turning angle in degrees.
     * @param da Turning angle in degrees.
     */
    public void setTurningAngle(double da) {
        state.da = da/180.0*Math.PI;
    }
    
    /**
     * Sets the length of a step.
     * @param len length of a step
     */
    public void setLength(double len) {
        state.l = len;
    }
    
    /**
     * Moves the turtle without drawing.
     */
    public void move() {
        state.x += Math.cos(state.a)*state.l;
        state.y += Math.sin(state.a)*state.l;
    }
    
    /**
     * Draws a line by moving the turtle.
     */
    public void draw() {
        if (Painter.size()==0) {
            throw new RuntimeException("No canvas is open! Aborting.");
        }
        int x = (int)Math.round(state.x);
        int y = (int)Math.round(state.y);
        move();
        Painter.drawLine(x, y, (int)Math.round(state.x), (int)Math.round(state.y));
    }
    
    /**
     * Turns the turtle to the right.
     */
    public void turnRight() {
        state.a += state.da;
    }
    
    /**
     * Turns the turtle to the left.
     */
    public void turnLeft() {
        state.a -= state.da;
    }
    
    /**
     * Turns right or left, depending on whether dir is true.
     * @param right if true, turn right, otherwise left.
     */
    public void turn(boolean right) {
        if (!right) {
            turnLeft();
        } else {
            turnRight();
        }
    }
   
    
    private class State {
        public double x;
        public double y;
        public double a;
        public double l=10.0;
        public double da = Math.PI/2;
        public State() {}
        public State(State s) {
            x = s.x;
            y = s.y;
            a = s.a;
            l = s.l;
            da = s.da;
        }
    }
}
