package ch.ksbg.fginfo;

import java.awt.Color;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.MouseInfo;
import java.awt.Point;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.imageio.ImageIO;
import javax.swing.ImageIcon;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.SwingUtilities;

/**
 * 
 * Class for easy access to drawing primitives, including
 * access to mouse coordinates and keystrokes.
 * 
 * @author Rolf Ingold, Ivo Blöchliger
 * 
 * 
 */



public class Painter {
    
    private static final ArrayList<Canvas> canvases = new ArrayList<>();    
    private static final ArrayList<Graphics> graphics = new ArrayList<>();
    private static int activeCanvas = -1;
    private static boolean doUpdates = true;
    
    
    private static void addCanvas(Canvas cvs) {
        canvases.add(cvs);
        activeCanvas = canvases.size()-1;
        graphics.add(cvs.getGraphics());
    }

    private static void check() {
        if (activeCanvas==-1) {
            throw new RuntimeException("Please first add a canvas with for example Painter.addCanvas(800,600);");
        }
    }

    /**
     * Suspend graphics updates.
     * Use this method to avoid flickering when drawing animations.
     * Do not forget to call {@link #resumeUpdates() resumeUpdates} to finally show the graphics.
     */
    public static void suspendUpdates() {
        doUpdates=false;
    }

    /**
     * Resume graphics updates.
     * See also the {@link #suspendUpdates() suspendUpdates} Method.
     */
    public static void resumeUpdates() {
        doUpdates=true;
        for (Canvas c : canvases) {
            c.update();
        }
    }

       
    
    /**
     * Open a new Canvas for drawing
     * @param width Width of the Canvas
     * @param height Height of the Canvas
     */
    public static void addCanvas(int width, int height) {
        addCanvas(new Canvas(width, height, String.format("Canvas %d",size())));
    }
    
    /**
     * Get the number of canvases
     * @return the number of canvases
     */
    public static int size() {
        return canvases.size();
    }
    
    /**
     * Get the width of the canvas
     * @return width of the canvas
     */
    public static int getWidth() {
        check();
        return canvases.get(activeCanvas).display.getWidth();
    }
    
    
    /**
     * Get the height of the canvas
     * @return height of the canvas
     */
    public static int getHeight() {
        check();
        return canvases.get(activeCanvas).display.getHeight();
    }
    
    
    /**
     * Select the canvas to use
     * @param index Index of the canvas (0,..,n-1)
     */
    public static void selectCanvas(int index) {
        // First report errors if index is out of bounds
        if (size()==0) {
            throw new RuntimeException("No canvas added yet! First add a canvas!");
        }
        if (index<0 || index>=size()) {
            throw new ArrayIndexOutOfBoundsException(String.format("Canvas with index %d does not exist. Index must be between 0 and %d.",index, size()-1));
        }
        activeCanvas = index;
    }

    /**
     * Fill the screen with white.
     */
    public static void clear() {
        clear(Color.WHITE);
    }

    /**
     * Fill the screen with a given Color.
     * The current color is not modified.
     * @param color Color to fill the screen with
     */
    public static void clear(Color color) {
        check();
        // Remeber current color
        Color savedColor = getColor();
        setColor(color);
        graphics.get(activeCanvas).fillRect(0,0,getWidth(), getHeight());
        // Restore color
        setColor(savedColor);
        canvases.get(activeCanvas).update();
    }

    /**
     * Get the current drawing Color
     * @return the currently used drawing Color
     */
    public static Color getColor() {
        check();
        return graphics.get(activeCanvas).getColor();
    }

    
    /**
     * Draw a line with the current color
     * @param x1 starting x-coordinate
     * @param y1 starting y-coordinate
     * @param x2 ending x-coordinate
     * @param y2 ending y-coordintate
     */
    public static void drawLine(int x1, int y1, int x2, int y2) {
        check();
        graphics.get(activeCanvas).drawLine(x1, y1, x2, y2);
        canvases.get(activeCanvas).update();
    }
    
    /**
     * Draws a circle with the current color
     * @param x x-coordinate of the center
     * @param y y-coordinate of the center
     * @param radius radius of the circle
     */
    public static void drawCircle(int x, int y, int radius) {
        check();
        graphics.get(activeCanvas).drawOval(x-radius, y-radius, 2*radius, 2*radius);
        canvases.get(activeCanvas).update();
    }
    
    /**
     * Fills a rectangle with the current color
     * @param x x-coordinate of the left border
     * @param y y-coordinate of the top border
     * @param width width of the rectangle
     * @param height height of the rectangle
     */
    public static void fillRect(int x, int y, int width, int height) {
        check();
        graphics.get(activeCanvas).fillRect(x, y, width, height);
        canvases.get(activeCanvas).update();
    }
    
    /**
     * Fills a circle with the current color
     * @param x x-coordinate of the center
     * @param y y-coordinate of the center
     * @param radius radius of the circle
     */
    public static void fillDisk(int x, int y, int radius) {
        check();
        graphics.get(activeCanvas).fillOval(x-radius, y-radius, 2*radius, 2*radius);
        canvases.get(activeCanvas).update();
    }
    
    /**
     * Fills the polygon described by the coordinates in the arrays x and y.
     * @param x Array of x-coordinates
     * @param y Array of y-corrdinates
     */
    public static void fillPolygon(int[] x, int[] y) {
        check();
        if (x.length!=y.length) {
            throw new IllegalArgumentException(String.format("Length of x-Array is %d, not equal to length of y-Array %d",x.length,y.length));
        }
        graphics.get(activeCanvas).fillPolygon(x, y, x.length);
        canvases.get(activeCanvas).update();
    }
    
    /**
     * Draws a String with the current color
     * @param text String to be drawn
     * @param x starting x-coordinate of the text
     * @param y baseline of the text
     */
    public static void drawString(String text, int x, int y) {
        check();
        graphics.get(activeCanvas).drawString(text, x, y);
        canvases.get(activeCanvas).update();
    }
    
    /**
     * Draws a String with the current color
     * @param text String to be drawn
     * @param x starting x-coordinate of the text
     * @param y baseline of the text
     * @param pt font-size in points
     */
    public static void drawString(String text, int x, int y, int pt) {
        check();
        graphics.get(activeCanvas).setFont(new Font(Font.MONOSPACED,Font.PLAIN,pt));
        graphics.get(activeCanvas).drawString(text, x, y);
        canvases.get(activeCanvas).update();
    }
    
    /**
     * Set the current color for the current Canvas. See {@link java.awt.Color}.
     * @param clr color to be set
     * 
     */
    public static void setColor(Color clr) {
        check();
        graphics.get(activeCanvas).setColor(clr);
    }
    
    /**
     * Get the RGB-value of a pixel
     * @param x x-coordinate of the pixel 
     * @param y y-coordinate of the pixel
     * @return rgb-value as an integer (i.e. b | (g &lt;&lt; 8) | (r &lt;&lt; 16)
     */
    public static int getRGB(int x, int y) {
        check();
        return canvases.get(activeCanvas).image.getRGB(x, y) & 0xffffff;
    }
    
    /**
     * Set a pixel to the given rgb value (i.e. b | (g &lt;&lt; 8) | (r &lt;&lt; 16)
     * You may also use 0x000000 to 0xffffff, for example red is 0xff0000.
     * @param x x-coordinate of the pixel 
     * @param y y-coordinate of the pixel 
     * @param rgb rgb-value as an integer (i.e. b | (g &lt;&lt; 8) | (r &lt;&lt; 16)
     */
    public static void setRGB(int x, int y, int rgb) {
        check();
        canvases.get(activeCanvas).image.setRGB(x, y, rgb);
        canvases.get(activeCanvas).update();
    }
    
    /**
     * Wait for a number of miliseconds.
     * @param duration number of miliseconds to wait.
     */
    public static void pause(int duration) {
        try {
            Thread.sleep(duration);
        } catch (InterruptedException exc) {
        }
    }
    
    /**
     * Get the position of the mouse over the current canvas.
     * @return The position of the mouse as a {@link java.awt.Point} object.
     */
    public static Point getMousePoint() {
        check();
        Point p = MouseInfo.getPointerInfo().getLocation();
        SwingUtilities.convertPointFromScreen(p,canvases.get(activeCanvas).display);
        return p;
    }
    
    /**
     * Get the x-coordinate of the mouse over the current canvas.
     * @return x-coordinate of the mouse
     */
    public static int getMouseX() {
        check();
        return getMousePoint().x;
    }

    /**
     * Get the y-coordinate of the mouse over the current canvas.
     * @return y-coordinate of the mouse
     */
    public static int getMouseY() {
        check();
        return getMousePoint().y;
    }
    
    /**
     * Get the current state of the Mouse Button (0 means no button is pressed).
     * @return Index of pressed mouse button (0 if none).
     */
    public static int getMouseButton() {
        check();
        return canvases.get(activeCanvas).mouseButtonState;
    }
    
    /**
     * Get a the keycode of the currently pressed key (0 if none is pressed).
     * @return keycode
     */
    public static int getPressedKey() {
        check();
        return  canvases.get(activeCanvas).keyPressed;
    }
    
    /**
     * Query if there are typed characters in the buffer.
     * @return true if there are characters in the buffer.
     */
    public static boolean isKeyTyped() {
        check();
        return canvases.get(activeCanvas).keyStrokes.size()>0;
    }
    
    /**
     * Get the oldest typed character in the buffer and remove it from the buffer.
     * If no character is in the buffer, the method returns '\0' (different from '0'!)
     * @return the oldest character or '\0' if none available.
     */
    public static char getKeyTyped() {
        check();
        if (isKeyTyped()) {
            return canvases.get(activeCanvas).keyStrokes.remove();
        }
        return 0;
    }
    
    /**
     * Get a String of all typed, not yet consumed characters (possibly empty).
     * This operation empties the buffer.
     * @return String of typed characters (possibly empty).
     */
    public static String getKeysTyped() {
        check();
        StringBuilder sb = new StringBuilder();
        while (!canvases.get(activeCanvas).keyStrokes.isEmpty()) {
            sb.append(canvases.get(activeCanvas).keyStrokes.remove());
        }
        return sb.toString();
    }
    
    /**
     * Wait for a key to be pressed, return the char.
     * 
     * Note: The implementation of this method is ugly and wastes resources!
     * @return char corresponding to the key pressed.
     */
    public static char waitForKey() {
        check();
        while (!isKeyTyped()) {
            pause(20);
        }
        return getKeyTyped();
    }
    
    /**
     * Wait for MouseButton to be pressed (or released).
     * 
     * Note: The implementation of this method is ugly and wastes resources!
     * @return The mask of the button pressed (or released).
     */
    public static int waitForMouseButton() {
        check();
        int button = 0;
        while ((button=getMouseButton())==0) {
            pause(10);
        }
        while (getMouseButton()!=0) {
            pause(20);
        }
        return button;
    }
    
    /**
     * Closes the current canvas.
     */
    public static void closeActiveCanvas() {
        check();
        graphics.remove(activeCanvas);
        canvases.get(activeCanvas).close();
        canvases.remove(activeCanvas);
        if (activeCanvas>=canvases.size()) {
            activeCanvas = canvases.size()-1;
        }
    }
    
    /**
     * Closes all canvases and terminates the program
     */
    public static void close() {
        for (Canvas c : canvases) {
            c.close();
        }
        System.exit(0);
    }
    
    /**
     * Save the current image as a PNG file. 
     * Warning: Existing files will be replaced.
     * @param fileName The filename to store the image.
     */
    public static void saveAsPNG(String fileName) {
        check();
        canvases.get(activeCanvas).save(fileName);
    }
    
    /**
     * Load an image file and display it at the top left.
     * @param fileName The filename to load the image from.
     */
    public static void loadFile(String fileName) {
        check();
        canvases.get(activeCanvas).load(fileName, 0,0);
    }
    
    /**
     * The actual Window object.
     */
    private static class Canvas {
    
        private final JLabel display;
        private final JFrame frame;
        private final Graphics graphics;
        private int mouseButtonState = 0;
        private final LinkedList<Character> keyStrokes = new LinkedList<>();
        private int keyPressed = 0;
        private final BufferedImage image;

        public Canvas(int width, int height, String title) {
            image = new BufferedImage(width, height, BufferedImage.TYPE_INT_RGB);
            display = new JLabel(new ImageIcon(image));
            frame = new JFrame();
            frame.setTitle(title);
            frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
            frame.setContentPane(display);
            frame.pack();
            graphics = image.createGraphics();
            graphics.fillRect(0, 0, width, height);
            graphics.setColor(Color.BLACK);
            display.addMouseListener(new MouseListener() {
                @Override
                public void mouseClicked(MouseEvent e) {
                }

                @Override
                public void mousePressed(MouseEvent e) {
                    mouseButtonState = e.getButton();
                }

                @Override
                public void mouseReleased(MouseEvent e) {
                    mouseButtonState = 0;
                }

                @Override
                public void mouseEntered(MouseEvent e) {
                }

                @Override
                public void mouseExited(MouseEvent e) {
                }
                
            });
            frame.addKeyListener(new KeyListener() {

                @Override
                public void keyTyped(KeyEvent e) {
                    keyStrokes.add(e.getKeyChar());
                }

                @Override
                public void keyPressed(KeyEvent e) {
                    keyPressed = e.getKeyCode();
                }

                @Override
                public void keyReleased(KeyEvent e) {
                    keyPressed = 0;
                }
            });
            frame.setVisible(true);
        }

        public Graphics getGraphics() {
            return graphics;
        }

        public void update() {
            if (doUpdates) {
                display.repaint();
            }
        }
        
        public void close() {
            frame.dispose();
        }

        private void save(String fileName) {
            try {
                ImageIO.write(image, "png", new File(fileName));
            } catch (IOException ex) {
                Logger.getLogger(Painter.class.getName()).log(Level.SEVERE, null, ex);
            }
        }

	private void load(String fileName, int left, int top) {
	    try {
		graphics.drawImage(ImageIO.read(new File(fileName)), left, top, null);
	    } catch (IOException ex) {
                Logger.getLogger(Painter.class.getName()).log(Level.SEVERE, null, ex);
            }
	}

    }

    
}
