package ch.ksbg.fginfo;

import java.util.Arrays;


/**
 * Some basic Matrix Operations, aimed at manipulating and transforming Polygondata.
 * @author Ivo Blöchliger, ivo.bloechliger@ksbg.ch
 */
public class Matrix {

    /**
     * Gibt eine Matrix gleicher Grösse zurück, aber
     * mit double Einträgen anstatt int.
     * @param mat int-Matrix
     * @return gleiche Matrix mit double-Einträgen
     */
    public static double[][] toDouble(int[][] mat) {
        double[][] res = new double[mat.length][mat[0].length];
        for (int i=0; i<mat.length; i++) {
            for (int j=0; j<mat[i].length; j++) {
                res[i][j] = mat[i][j];  // int->double, kein cast nötig.
            }
        }
        return res; 
    }


    /**
     * Gibt eine Matrix gleicher Grösse zurück, aber
     * mit int Einträgen anstatt double.
     * @param mat double-Matrix
     * @return "gleiche" Matrix mit gerundeten int-Einträgen
     */    
    public static int[][] toInt(double[][] mat) {
        int[][] res = new int[mat.length][mat[0].length];
        for (int i=0; i<mat.length; i++) {
            for (int j=0; j<mat[i].length; j++) {
                res[i][j] = (int)Math.round(mat[i][j]);  // long->int, cast ist nötig.
            }
        }
        return res; 
    }

    /**
     * Generiert ein neues n x m Array.
     * @param n Anzahl Zeilen
     * @param m Anzahl Spalten
     * @return n x m double-Matrix, mit 0.0 gefüllt.
     */
    public static double[][] newMat(int n, int m) {
        return new double[n][m];
    }
    
    /**
     * Erstellt eine echte Kopie der Matrix.
     * @param mat Original-Matrix
     * @return echte Kopie
     */
    public static double[][] copy(double[][] mat) {
        double[][] res = new double[mat.length][];
        for (int i=0; i<mat.length; i++) {
            res[i] = Arrays.copyOf(mat[i],mat[i].length);
        }
        return res;
    }
    
    /**
     * Erstellt eine 2 x 2 Rotationsmatrix gemäss Formel.
     * Der Winkel wird in Radianten angenommen
     * @param angle Drehwinkel in Radianten
     * @return 2 x 2 Rotationsmatrix
     */
    public static double[][] rotMat(double angle) {        
        return new double[][] {{Math.cos(angle), -Math.sin(angle)},{Math.sin(angle), Math.cos(angle)}};
    }
    
    /**
     * Erstellt eine 2 x 2 Skalierungsmatrix mit
     * Faktor lambda.
     * @param lambda Streckfaktor
     * @return Matrix für Streckung am Ursprung mit Faktor lambda.
     */
    public static double[][] scaleMat(double lambda) {
        return new double[][] {{lambda, 0.0},{0.0, lambda}};
    }

    /**
     * Matrix-Matrix Multiplikation.
     * @param a erste Matrix (n x m)
     * @param b zweite Matrix (m x k)
     * @return Produkt a * b als n x k Matrix
     */
    public static double[][] mulMat(double[][] a, double[][] b) {
        double[][] c = newMat(a.length, b[0].length);
        for (int i=0; i<a.length; i++) {
            for (int j=0; j<b[0].length; j++) {
                for (int l=0; l<b.length; l++) {
                    c[i][j]+=a[i][l]*b[l][j];
                }
            }
        }
        return c;
    }

    /**
     * Alle Spalten einer Matrix verschieben.
     * @param mat Originalmatrix (n x m)
     * @param vec Verschiebungsvektor (n x 1)
     * @return neue Matrix mit veränderten Spalten
     */
    public static double[][] translate(double[][] mat, double[][] vec) {
        double[][] res = copy(mat);
        for (int i=0; i<mat.length; i++) {
            for (int j=0; j<mat[0].length; j++) {
                res[i][j]+=vec[i][0];
            }
        }
        return res;
    }

    /**
     * Umwandlung einer Matrix in einen String.
     * Einfachen Ausgabe, z.B. wenn mat eine Matrix ist:
     * <code>System.out.println(Matrix.toString(mat))</code>
     * @param mat Matrix
     * @return mehrzeilige String-Darstellung
     */
    public static String toString(double[][] mat) {
        StringBuilder sb = new StringBuilder();
        for (int i=0; i<mat.length; i++) {
            if (i==0) {
                sb.append("/ ");
            } else if (i==mat.length-1) {
                sb.append("\\ ");
            } else {
                sb.append("| ");
            }
            for (int j=0; j<mat[i].length; j++) {
                sb.append(String.format("%8.3f ",mat[i][j]));
            }
            if (i==0) {
                sb.append("\\");
            } else if (i==mat.length-1) {
                sb.append("/");
            } else {
                sb.append("|");
            }            
            sb.append("\n");
        }
        return sb.toString();
    }

}
