package nim;

import java.util.Arrays;
import java.util.Iterator;

/**
 * Class for storing and manipulating a Nim-Game Situation
 */

public class Situation{
    private int[] rows;
    private int[] init;
    private int[] baseProduct;
    private int[] bases;
    
    private int numConfig;
    private int total;
    
    /**
     * Initialize to a given array
     * @param r 
     */
    public Situation (int[] r) {
        rows = Arrays.copyOf(r, r.length);
        setInit();
    }
    
    /**
     * Initialize to a given number of rows, filled with 1,3,5,....
     * @param r 
     */
    public Situation (int r) {
        rows = new int[r];
        for (int i=0; i<r; i++) {
            rows[i] = 2*i+1;
        }
        setInit();
    }
    
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        for (int i=0; i<rows.length; i++) {
            sb.append(String.format("[r=%d, n=%d]  ", i, rows[i]));
            for (int j=0; j<rows[i];j++) {
                sb.append('|');
            }
            sb.append(String.format("%n"));
        }
        return sb.toString();
    }

    /**
     * Revert to the original situation
     */
    public void reset() {
        rows = Arrays.copyOf(init, init.length);
        computeTotal();
    }
    
    /**
     * Compute a unique number representing this configuration using
     * a mixed bases.
     * @return 
     */
    public int getConfigNumber() {
        int n = 0;
        for (int i=0; i<rows.length; i++) {
            n += rows[i]*baseProduct[i];
        }
        return n;
    }
    
    /**
     * Set the configuration from the unique number representing a situation.
     * Inverse function to getConfigNumber()
     */
    public void setConfigFromNumber(int n) {
        for (int i=0; i<rows.length-1; i++) {
            rows[i] = n % bases[i+1];
            n /= bases[i+1];
        }
        rows[rows.length-1] = n;
        computeTotal();
    }
    
    /**
     * Get the number of matches in row 
     * @param r
     */
    public int getRow(int r) {
        return rows[r];
    }
    
    /**
     * Get an Array representing all rows
     * @return 
     */
    public int[] getRows() {
        return Arrays.copyOf(rows, rows.length);
    }
    
    /**
     * Get the number of rows
     * @return 
     */
    public int getSize() {
        return rows.length;
    }
    
    /**
     * Number of possible configurations
     */
    public int getNumConfigs() {
        return numConfig;
    }
    
    /**
     * Remove a number of matches from a row
     * @param row
     * @param num 
     */
    public void remove(int row, int num) {
        if (num<1 || num>rows[row]) {
            throw new IllegalArgumentException(String.format("Not allowed to remove %d matches from row %d containing %d matches", num, row, rows[row]));
        }
        rows[row] -= num;
        total -= num;
    }
    
    /**
     * Have all matches been taken away?
     * @return 
     */
    public boolean gameOver() {
        return total == 0;
    }
    
    /**
     * Get an Iterator over all situations than can be attained from this situation
     * @return 
     */
    public Iterator<Situation> iterator() {
        return new SituationIterator();
    }
    
    private void computeTotal() {
        total = 0;
        for (int i=0; i<rows.length; i++) {
            total += rows[i];
        }
    }
    
    
    /**
     * Get all situations that can be attained from this situation
     */
    public class SituationIterator implements Iterator<Situation> {
        private int r=0;
        private int n=0;
        
        public SituationIterator() {
            while (r<rows.length && rows[r]==0) { r++; }
            n = 1;
        }
        
        @Override
        public boolean hasNext() {
            return (r<rows.length);
        }

        @Override
        public Situation next() {
            int[] s = Arrays.copyOf(rows, rows.length);
            s[r] -= n;
            if (n<rows[r]) {
                n++;
            } else {
                r++;
                while (r<rows.length && rows[r]==0) { r++; }
                n=1;
            }
            return new Situation(s);
        }

        @Override
        public void remove() {
            throw new UnsupportedOperationException("Not supported.");
        }
        
    }
        
    
    
    private void setInit() {
        init = Arrays.copyOf(rows, rows.length);
        baseProduct = new int[rows.length];
        bases = new int[rows.length];
        baseProduct[0] = 1;
        bases[0] = 1;
        for (int i=1; i<rows.length; i++) {
            bases[i] = rows[i-1]+1;
            baseProduct[i] = baseProduct[i-1]*(rows[i-1]+1);
        }
        numConfig = baseProduct[rows.length-1]*(rows[rows.length-1]+1);
        computeTotal();
    }

}
