/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package hermitboxes;

import java.util.Arrays;
import java.util.Random;

/**
 *
 * @author ivo
 */
public class Field {
    private HermitBoxes game;
    
    private long[] fields;
    private int[][] contents;
    private long[][] masks;
    private int[] movePositions;
    private int[] moveDirections;
    private int[] moveColors;
    private long[] moveBackups;
    private int turnCounter=0;
    
    private int[] bestMove = new int[4];
    
    public Field(HermitBoxes g) {
        if (g.width*g.height>64) {
            throw new RuntimeException("Field too big, maximum of 64 positions!");
        }
        game = g;
        fields = new long[g.colors+1];
        contents = new int[g.width][g.height];
        movePositions = new int[g.width*g.height];
        moveColors = new int[g.width*g.height];
        moveDirections = new int[g.width*g.height];
        moveBackups = new long[g.width*g.height];
        masks = new long[g.width][g.height];
        makeMasks();
    }

    private void makeMasks() {
        for (int x=0; x<game.width; x++) {
            for (int y=0; y<game.height; y++) {
                for (int a=x-1; a<=x+1; a++) {
                    if (a>=0 && a<game.width) {
                        for (int b=y-1; b<=y+1;b++) {
                            if (b>=0 && b<game.height) {
                                masks[x][y] |= (1L << (a+b*game.width));
                            }
                        }
                    }
                }
            }
        }
    }
    
    // Return true if we have a winning situation
    public boolean makeMove(int x, int y, int c, int d) {
        // Check if move is legal. This is only useful for development.
//        if (!isMoveLegal(x, y, c, d, true)) {
//            throw new IllegalArgumentException("see message to stdout");
//        }
        
        int a = x + ((d==1)?1:0);
        int b = y + ((d==2)?1:0);
        
        // Now execute the move and store history
        contents[x][y] = c;
        fields[0] |= 1L << (x+y*game.width);
        moveColors[turnCounter] = c;
        movePositions[turnCounter] = x+y*game.width;
        moveDirections[turnCounter] = d;
        moveBackups[turnCounter] = fields[c];
        fields[c] |= masks[x][y];
        if (d>0) {
            contents[a][b] = c;
            fields[c] |= masks[a][b];
            fields[0] |= 1L << (a+b*game.width);
        }
        turnCounter++;
        return isBoardFull();
    }
    
    public void undoMove() {
        if (turnCounter>0) {
            turnCounter--;
            int x = movePositions[turnCounter] % game.width;
            int y = movePositions[turnCounter] / game.width;
            int c = moveColors[turnCounter];
            int d = moveDirections[turnCounter];
            int a = x + ((d==1)?1:0);
            int b = y + ((d==2)?1:0);
            contents[x][y] = 0;
            fields[0] ^= 1L << (x+y*game.width);
            if (d>0) {
                contents[a][b] = 0;
                fields[0] ^= 1L << (a+b*game.width);
            }
            fields[c] = moveBackups[turnCounter];
            
        } else {
            throw new RuntimeException("No further undo possible!");
        }
    }

    public boolean isMoveLegal(int x, int y, int c, int d, boolean show) {
        if (x<0 || y<0 || x>=game.width || y>=game.height || c<1 || c>game.colors || d<0 || d>2) {
            if (show) {
                System.out.format("Parameters out of range: x=%d, y=%d, c=%d, d=%d%n",x,y,c,d);
            }
            return false;
        }
        int a = x + ((d==1)?1:0);
        int b = y + ((d==2)?1:0);
        if (a>=game.width || b>=game.height) {
            if (show) {
                System.out.format("Parameters out of range: x=%d, y=%d, c=%d, d=%d => a=%d, b=%d%n",x,y,c,d,a,b);
            }
            return false;
        }
        if (contents[x][y]!=0) {
            if (show) {
                System.out.format("Field x=%d, y=%d not empty (used with color %d)%n",x,y, contents[x][y]);
            }
            return false;
        }
        if ((fields[c] & (1L << (x+y*game.width)))>0) {
            if (show) {
                System.out.format("Field x=%d, y=%d adjacent to color %d%n",x,y,c);
            }
            return false;
        }
        if (d>0 && contents[a][b]!=0) {
            if (show) {
                System.out.format("Field a=%d, b=%d not empty (used with color %d)%n",a,b, contents[a][b]);
            }
            return false;
        }
        if (d>0 && (fields[c] & (1L << (a+b*game.width)))>0) {
            if (show) {
                System.out.format("Field a=%d, b=%d adjacent to color %d%n",a,b,c);
            }
            return false;
        }
        return true;
    }
    
    public boolean isBoardFull() {
        for (int color=1; color<=game.colors; color++) {
            if (Long.bitCount(fields[color] | fields[0])<game.width*game.height) {
                return false;
            }
        }
        return true;
    }
    
    public int getColor(int x, int y) {
        return contents[x][y];
    }
    
    public int getPlayer() {
        return turnCounter%2;
    }
    
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("   ");
        StringBuilder sep = new StringBuilder("   ");
        for (int x=0; x<game.width;x++) {
            sep.append("+---");
            sb.append(String.format("  %d ",x));
        }
        sb.append(String.format("%n"));
        sep.append(String.format("+%n"));
        sb.append(sep);
        for (int y=0; y<game.height; y++) {
            sb.append(String.format(" %d ",y));
            for (int x=0; x<game.width;x++) {
                if (contents[x][y]>0) {
                    sb.append(String.format("| %d ",contents[x][y]));
                } else {
                    sb.append("|   ");
                }
            }
            sb.append(String.format("|%n"));
            sb.append(sep);
        }
        
        for (int color = 0; color<=game.colors;color++) {
            sb.append(String.format("Color %d",color));
            for (int w=0; w<game.width*4-3; w++) {
                sb.append(" ");
            }
            sb.append(String.format("%n"));
            sb.append(sep);
            for (int y=0; y<game.height; y++) {
                sb.append(String.format(" %d ",y));
                for (int x=0; x<game.width;x++) {
                    if ((fields[color] & (1L << (x+y*game.width)))>0) {
                        sb.append(String.format("| %d ",color));
                    } else {
                        sb.append("|   ");
                    }
                }
                sb.append(String.format("|%n"));
                sb.append(sep);
            }
        }
        String[] lines = sb.toString().split("\n");
        sb = new StringBuilder();
        for (int i=0; i<game.height*2+2; i++) {
            for (int c=0; c<=game.colors+1;c++) {
                sb.append(lines[i+c*(game.height*2+2)]);
                sb.append("   ");
            }
            sb.append(String.format("%n"));
        }
        return sb.toString();
    }
    
    public void chooseMove(int timeLimit) {
        for (int depth = 1; depth<20; depth+=1) {
            long startTime = System.nanoTime();
            double v = evaluate(depth, true);
            startTime = (System.nanoTime()-startTime)/1000000;            
            System.out.format("Depth %d -> v=%f time %dms%n",depth, v, startTime);
            if (v==1.0 || v==-1.0 || startTime > timeLimit) {
//                System.out.format("Finishing with v=%f: %s  after %dms%n",v, Arrays.toString(bestMove),startTime);
                makeMove(bestMove[0], bestMove[1], bestMove[2], bestMove[3]);
                return;
            }
        }
        // Random move
        makeMove(bestMove[0], bestMove[1], bestMove[2], bestMove[3]);
    }
    
    
     public double evaluate(int maxDepth, boolean first) {
        if (isBoardFull()) {
            return -1.0; // We loose
        }
        if (maxDepth==0) {
            return 0.0; // no idea...
        }
        int numMoves = 0;
        Random r = new Random();
        double bestv = -1.0;
        // Try all moves
        for (int x=0; x<game.width; x++) {
            for (int y=0; y<game.height; y++) {
                if (contents[x][y]==0) {
                    for (int c=1; c<=game.colors; c++) {
                        for (int d=0; d<3; d++) {
                            if (isMoveLegal(x, y, c, d, false)) {
                                numMoves++;
                                makeMove(x, y, c, d);
                                double v = -evaluate(maxDepth-1, false);
                                undoMove();
                                if (v>bestv) {
                                    numMoves=1;
                                    bestv = v;
                                    if (first) {
                                        bestMove[0] = x;
                                        bestMove[1] = y;
                                        bestMove[2] = c;
                                        bestMove[3] = d;
                                    }
                                    //System.out.format("Winning move at d=%d (numMoves=%d) with (%d, %d, %d, %d)%n%s",maxDepth, numMoves,x,y,c,d,this);
                                    if (v==1.0) {
                                        return 1.0;
                                    }
                                }
                                if (first && v==bestv && r.nextInt(numMoves)==0) {
                                    bestMove[0] = x;
                                    bestMove[1] = y;
                                    bestMove[2] = c;
                                    bestMove[3] = d;
                                }
                            }
                        }
                    }
                }
            }
        }
        if (first) {
            System.out.format("at Depth %d found %d moves with v=%f%n",maxDepth, numMoves, bestv);
        }
        return bestv;
    }
}
