package viergewinnt;

import java.util.HashMap;
import java.util.Random;

/**
 * A strategy implementing alpha,beta pruning, iterative deepening and storing
 * of computed evaluations in a hash-table.
 * 
 * It still lacks dynamic move-ordering based on previous evaluations.
 */
public class Strategy {
    
    private AbstractField field;
    /**
     * For move-ordering (not dynamic for now)
     */
    private int[] order;
    /**
     * The best move found
     */
    private int bestPosition;
    /**
     * Counter to count how many times the stored situations have been reused.
     */
    private int reuse;

    /**
     * Hashmap to store evaluations (used only when the field is a BitField
     */
    private HashMap<Long, Double> hashMap = new HashMap<>();
    
    
    public Strategy(AbstractField f) {
        field = f;
        order = new int[field.width];
    }
    
    /**
     * The time limit is exceeded. The computation stops once an evaluation has
     * taken more time than timeLimit.
     * @param timeLimit 
     */
    public void makeMove(int timeLimit) {
        orderedIterativeDeepening(timeLimit);
        field.makeMove(bestPosition);
    }
    
    /**
     * It sets the instance variable bestPosition, containing the best move
     * Note: The evaluation is only stored, when the situation is completely evaluated.
     * If the evaluation is stoped because alpha>=beta, the evaluation is not stored (it might actually be highter)
     */
    public double alphaBetaPruning(int maxDepth, boolean first, double alpha, double beta) {
        if (field instanceof BitField) {
            Double r = hashMap.get(((BitField)field).getConfig());
            if (r!=null) {
                reuse++;
                return r;
            }
        }      
        boolean found = false;
        double mul = (field.getPlayer()==0) ? 1.0 : -1.0;
        for (int i=0; i<field.width; i++) {
            int w = order[i];
            if (field.isOpen(w)) {
                if (first && !found) {
                    bestPosition = w;
                    found = true;
                }
                field.makeMove(w);
                double v;
                if (field.checkWin(1-field.getPlayer())) {
                    //System.out.format("Winning position at %d%n%s",maxDepth,field);
                    v = 1.0;
                } else if (field.checkDraw()) {
                    v = 0.0;
                } else {
                    if (maxDepth==0) {
                        v = mul*field.evaluatePlayerZero();
                    } else {
                        v = -alphaBetaPruning(maxDepth-1, false, -beta, -alpha);
                    }
                }
                field.undoMove();

                if (v>alpha) {
                    if (first) {
                        bestPosition = w;
                    }
                    alpha = v;
                }
                if (beta<=alpha) {
                    // This is just an underestimation of the value of this position, do not store
                    //if (field instanceof BitField) {
                    //    hashMap.put(((BitField)field).getConfig(), alpha);
                    //}
                    return alpha;
                }                
            }
        }
        if (field instanceof BitField) {
            hashMap.put(((BitField)field).getConfig(), alpha);
        }
        return alpha;
    }
    
    /**
     * The order is fixed, from center to border, with random permutation 
     * of symmetric columns
     * 
     * The method will exceed the timeLimit and stop as soon as the exploration
     * needed more time or the complete tree has been explored.
     * @param maxMilis
     * @return value of the position
     */
    private double orderedIterativeDeepening(int maxMilis) {
        double v=0.0;
        int lastBest;
        int n=0;
        Random r = new Random();
        for (int i=0;i<=field.width/2; i++) {
            order[n++] = field.width/2-i;
            if (i>0 && i+field.width/2<field.width) {
                order[n++]=i+field.width/2;
                if (r.nextBoolean()) {
                    order[n-2] = field.width/2+i;
                    order[n-1] = field.width/2-i;
                }
            }
        }
        for (int d = 0; d<field.width*field.height-field.turnCounter; d++) {
            hashMap.clear();
            reuse = 0;
            lastBest = bestPosition;
            long startTime = System.nanoTime();
            v = alphaBetaPruning(d,true,-1.0, 1.0);
            System.out.format("hashMap has %d entries, producing %d hits%n",hashMap.size(), reuse);
            startTime = (System.nanoTime()-startTime)/1000000;
            System.out.format("Depth %d, value %f, decision %d, %dms%n",d,v,bestPosition, startTime);
            if (startTime > maxMilis || v==1.0 || v==-1.0) {
                if (v==-1.0) {
                    bestPosition = lastBest;
                }
                return v;
            }
            
        }
        return v;
    }


    
}
