package viergewinnt;


/**
 * Connect-4 field implemented in a single 64-Bit long
 */
public class BitField extends AbstractField {
   
    /**
     * Bit fields for players
     * The first two are the actual bitfields
     * The second pair are the mirrored bitfields.
     */
    long[] players = new long[] {0L, 0L, 0L, 0L};
    
    /**
     * Endbit of each column
     */
    int heights[];
    /**
     * Board height+1,
     * Board height+2
     */
    final int H1, H2;
    /**
     * Bitmasks for the bottom row, top row and actual playing field
     */
    final long BOTTOM, TOP, FIELD;
    
    /**
     * Columns played
     */
    private int[] history;
        
    /**
     * width * (height+1) must be smaller than or equal to 64
     * @param w
     * @param h 
     */
    public BitField(int w, int h) {
        super(w,h);
        if (w*(h+1)>64) {
            throw new RuntimeException("This field will not fit in 64bits!");
        }
        H1 = h+1;
        H2 = h+2;
        heights = new int[w];
        for (int i=0; i<w; i++) {
            heights[i] = i*(H1);
        }
        BOTTOM = ((1L << w*H1) - 1)/((1L << H1)-1);
        TOP = BOTTOM << height;
        FIELD = ((1L << w*H1)-1)^TOP;
        history = new int[width*height];
    }

    /**
     * Get the configuration of the  Board in a long
     * Returns the symmetric configuration if its number is smaller
     * @return 
     */
    public long getConfig() {
        long config = 2*players[0]+players[1]+BOTTOM;
        long symmetric = 2*players[2]+players[3]+BOTTOM;
        if (config <= symmetric) {
            return config;
        } else {
            return symmetric;
        }
    }
    
    /**
     * Makes a move (also for the symmetric situation)
     * @param column 
     */
    @Override
    public void makeMove(int column) {
        history[turnCounter] = column;
        players[(turnCounter&1)+2] |= 1L << (heights[column]-column*H1+(width-column-1)*H1); // Symmetric situation
        players[(turnCounter++)&1] |= 1L << heights[column]++;
    }

    /**
     * Undoes a move (also for the symmetric situation)
     */
    @Override
    public void undoMove() {
        // subexpressions are evaluated from left to right. Always.
        players[(--turnCounter)&1] ^= 1L << --heights[history[turnCounter]];
        players[(turnCounter&1)+2] ^= 1L << (heights[history[turnCounter]]-history[turnCounter]*H1+(width-history[turnCounter]-1)*H1); // Symmetric situation
    }

    @Override
    public boolean checkWin(int player) {
        long d = players[player] & (players[player] << 1); // vertical
        if ((d & (d<<2))!=0) {
            return true;
        }
        d = players[player] & (players[player] << H1); // horizontal
        if ((d & (d<<2*H1))!=0) {
            return true;
        }
        d = players[player] & (players[player] << H2); // diag NE
        if ((d & (d<<2*H2))!=0) {
            return true;
        }
        d = players[player] & (players[player] << height); // diag SE
        if ((d & (d<<2*height))!=0) {
            return true;
        }
        return false;
    }
    /**
     * Count the number of possible 4-in a row one can build using
     * existing coins plus empty spaces.
     * I don't believe this heuristic to be very good. Experiement with others.
     * @return 
     */
    @Override
    public double evaluatePlayerZero() {
        if (checkWin(0)) {
            return 1.0;
        }
        if (checkWin(1)) {
            return -1.0;
        }
        if (checkDraw()) {
            return 0.0;
        }
        long empty = ((1L << (width*H1))-1) ^ players[0] ^ players[1];
        // Possibilities to still make 4 using existing pieces
        int[] poss = new int[2];
        for (int i=0; i<2; i++) {
            long c = players[i]|empty; // Player plus empty
            // vertical
            long d = c & (c>>1);
            d = (d & (d>>2));
            long e = players[i];
            long f = e;
            for (int j=0; j<3; j++) {
                f = (f >> 1) & FIELD;
                e |= f;
            }
            poss[i] += Long.bitCount(e & d);
            // horizontal
            d = c & (c>>H1);
            d = d & (d>>2*H1);
            e = players[i] | (players[i] >> H1) | (players[i] >> 2*H1) | (players[i] >> 3*H1);
            poss[i] += Long.bitCount(e & d);
            // diag NE
            d = c & (c>>H2);
            d = d & (d>>2*H2);
            e = players[i];
            f = e;
            for (int j=0; j<3; j++) {
                f = (f >> H2) & FIELD;
                e |= f;
            }
            poss[i] += Long.bitCount(e & d);
            // diag SE
            d = c & (c>>height);
            d = d & (d>>2*height);
            e = players[i];
            f = e;
            for (int j=0; j<3; j++) {
                f = (f >> height) & FIELD;
                e |= f;
            }        
            poss[i] += Long.bitCount(d & e);
        }
        return ((double)(poss[0]-poss[1]))/(width*height*16);
    }

    @Override
    public int getField(int x, int y) {
        long mask = 1L << (y+x*H1);
        if ((players[0] & mask)!=0) {
            return 0;
        }
        if ((players[1] & mask)!=0) {
            return 1;
        }
        return -1;
    }

    @Override
    public boolean isOpen(int column) {
        return heights[column] < (column+1)*H1-1;  // precomputing these values might speed things up.
    }

}
