package viergewinnt;

/**
 * Abstract Field Class implementing some basic methods and
 * defining basic methods every playing field must have.
 */
public abstract class AbstractField {
    
    /**
     * Width and height of the playing field
     */
    public final int width, height;
    
    /**
     * Number of turns played
     */
    protected int turnCounter;
    
    
    public AbstractField(int w, int h) {
        width = w;
        height = h;
        turnCounter=0;
    }
    
    /**
     * Drop a coin in column
     * @param column 
     */
    public abstract void makeMove(int column);
    
    /**
     * Undo last move
     */
    public abstract void undoMove();
    
    /**
     * Check if player (0 or 1) has won
     * @param player
     * @return 
     */
    public abstract boolean checkWin(int player);
    
    /**
     * Heuristic to evaluate the quality of the position for player 0
     * @return value between -1.0 and +1.0
     */
    public abstract double evaluatePlayerZero();
    
    /**
     * What is on field x,y?
     * @param x
     * @param y
     * @return -1 (empty), 0 (player 1) or 1 (player 2)
     */
    public abstract int getField(int x,int y);
    
    /**
     * Checks if there can a coin be dropped into
     * @param column
     * @return 
     */
    public abstract boolean isOpen(int column);
    
    /**
     * Returns true if the board is full (and nobody won)
     * @return 
     */
    public boolean checkDraw() {
        if ((turnCounter==width*height) && (!checkWin(0)) && (!checkWin(1))) {
            return true;
        }
        return false;
        
    }
    
    /**
     * Returns the number of turns already played
     * @return 
     */
    public int getTurnCounter() {
        return turnCounter;
    }
    
    /**
     * Returns the player who plays next
     * @return 0 (player 1) or 1 (player 2)
     */
    public int getPlayer() {
        return turnCounter & 1;
    }

    /**
     * Build a string representing the current board
     * @return 
     */
    @Override
    public String toString() {
        StringBuilder sep = new StringBuilder();
        StringBuilder f = new StringBuilder();
        for (int w=0;w<width;w++) {
            sep.append("+---");
            f.append(String.format(" %2d ",w));
        }
        f.append(String.format("%n"));
        for (int w=0; w<width; w++) {
            f.append("+   ");
        }
        sep.append(String.format("+%n"));
        f.append(String.format("+%n"));
        String[] symbol = {"|   ", "| # ", "| . "};
        for (int h=height-1; h>=0; h--) {
            for (int w=0; w<width; w++) {
                f.append(symbol[getField(w,h)+1]);
            }
            f.append(String.format("|%n"));
            f.append(sep);
        }
        return f.toString();
    }
    
}