/** 
 * Die Graphendateien haben folgenden Syntax:
 * Erste Zeile: Anzahl Knoten (Ganzzahl n)
 * 
 * Danach Zeilen, deren Einträge durch genau 1 Leerschlag getrennt sind.
 * c i x y
 *   wobei i die Nummer eines Knotens ist (0..(n-1)) und x,y sind Kommazahlen für die Koordinaten des Knotens
 *
 * e i j [w]
 *   Spezifiziert eine ungerichtete Kante wobei i,j die Nummern der Knoten sind (0..(n-1)).
 *   Der letzte Eintrag ist optional (noch nicht implementiert) und enthält eine Kommazahl für das Kantengewicht (z.B. Länge).
 */


class Graph {
  public boolean adj[][];
  public float x[];
  public float y[];
  private int n;
  
  private float xmin, xmax, ymin, ymax; 
  
  // Graph ohne Kanten mit n Knoten initialisieren.
  public Graph(int n) {
    this.n = n;
    initializeData( );
  }
  
  // Anzahl Knoten erfragen.
  public int getN() {
    return n;
  }
  
  // Graph von Datei einlesen.
  Graph(String filename) {
    BufferedReader file = createReader(filename);
    try {
      // n einlesen
      int n = int(file.readLine());
      this.n = n;
      initializeData();
      while (true) {
        // Zeilen einlesen
         String line = file.readLine();
         if (line==null) break;
         // Weder Leerzeile noch Kommentar
         if (line.length()>0 && line.charAt(0)!='#') {
           String[] items = split(line," ");
           // Koordinatenangabe
           if (items[0].equals("c")) {
             int v = int(items[1]);
             x[v] = float(items[2]);
             y[v] = float(items[3]);
             //println("Vertex "+v+" at ("+x[v]+","+y[v]+")");
           } else if (items[0].equals("e")) {
             // Kantenangabe
             int v = int(items[1]);
             int u = int(items[2]);
             addEdge(u,v,false);
             if (items.length>3) {
               println("WARNING: Edge weights not implemented!");
             }
           }
         }
      }
    } catch (IOException e) { // Probleme mit der Datei "behandeln"...
      e.printStackTrace();
      System.exit(-1);
    }    
  }
  
  // Kante hinzufügen (muss evtl. für weitere Datenstrukturen angepasst werden.
  public void addEdge(int u, int v, boolean oriented) {
    //println("Adding edge "+u+" <-> "+v);
    adj[u][v] = true;
    if (!oriented) {
      adj[v][u] = true;
    }
  }
  
  // Berechnet die BoundingBox
  private void computeDrawSize() {
    xmin = (xmax = x[0]);
    ymin = (ymax = y[0]);
    for (float xx : x) {
      if (xx<xmin) xmin=xx;
      if (xx>xmax) xmax=xx;
    }
    for (float yy : y) {
      if (yy<ymin) ymin=yy;
      if (yy>ymax) ymax=yy;
    }
  }
  
  private float[] getTrans() {
    // Grösse des Graphen ermitteln
    computeDrawSize();
    // Streckfaktor ermitteln
    float scale = width/(xmax-xmin);
    float s2 = height/(ymax-ymin);
    if (s2<scale) scale = s2;
    scale*=0.95;
    //println(scale);
    // Verschiebung ermitteln
    float xoffset = 0.025*width-xmin*scale;
    float yoffset = 0.025*height-ymin*scale;
    //println(xoffset);
    //println(yoffset);
    return new float[] {scale, xoffset, yoffset};
  }
  
  public void draw() {
    float[] trans = getTrans();
    for (int i=0; i<n; i++) {
      // Knoten Koordinaten
       float cx = x[i]*trans[0]+trans[1];
       float cy = y[i]*trans[0]+trans[2];
       for (int j=i+1; j<n; j++) {
         // Kanten zeichnen
         if (adj[i][j]) {
           float jx = x[j]*trans[0]+trans[1];
           float jy = y[j]*trans[0]+trans[2];
           line(cx,cy,jx,jy);
         }
       }
      // Knoten zeichnen
       ellipse(cx,cy,10,10);
    }
  }

  // Speicher allozieren
  private void initializeData() {
    adj = new boolean[n][n];
    x = new float[n];
    y = new float[n];
  }
  
  public int degree(int v) {
    int deg = 0;
    for (int i=0; i<n; i++) {
      if (adj[v][i]) deg++;
    }
    return deg;
  }
  
  // Returns a cycle, without the last vertex  
  private Integer[] subpath(boolean[][] available, int start) {
      ArrayList<Integer> res = new ArrayList<Integer>();
      int cur = start;
      res.add(start);
      boolean finished = false;
      while (!finished) {
        finished = true;
         for (int j=0; j<n; j++) {
            if (available[cur][j]) {
              available[cur][j]=false;
              available[j][cur]=false;
              cur = j;
              res.add(j);
              finished = false;
              break;
            }
         }
      }
      // Remove last element, in case it is a cycle
      if (res.get(0)==res.get(res.size()-1)) {
        res.remove(res.size()-1);
      }
      return res.toArray(new Integer[0]);
  }
  
  public int[] eulerpath() {
    int start = 0;
    int odds = 0;
    // Available edges
    boolean[][] av = new boolean[n][n];
    int m = 0;
    // check degrees, find start vertex
    for (int i=0; i<n; i++) {
      for (int j=0; j<n; j++) {
        av[i][j] = adj[i][j];
      }
      int d = degree(i);
      m+=d;
       if (d % 2 == 1) {
         start = i;
         odds++;
       }
    }
    m/=2;
    // Too many odd vertices, return null
    if (odds>2) {
      return null;
    }
    // Result Array
    Integer[] res = new Integer[1];
    // Starting node, covered edges, where to insert next subcycle
    res[0] = start;
    int covered = 0;
    int insertAt = 0;
    if (odds>0) {
      res = new Integer[0];
    }
    while (true) {
      // Get subcycle
      Integer[] path = subpath(av, start);
      print("Insert at position "+insertAt+" (node "+start+") subpath:");
      println(path);
      // splice to insert an Array
      res = (Integer[])splice(res,path, insertAt);
      println(res);
      start = -1;
      covered+=path.length;      
      for (int i=0; i<covered; i++) {
         for (int j=0; j<n; j++) {
           if (av[res[i]][j]) {
             start = res[i];
             insertAt = i;
             break;
           }
         }
      }
      if (start==-1) break;
    }
    // Convertsion to int[]
    int[] result=new int[res.length];
    for (int i=0; i<res.length; i++) result[i]=res[i];
    return result;
  }
  
  private float[] unitv(int a, int b) {
    float[] r = new float[2];
    if (a==b) return r;
    r[0] = x[b]-x[a];
    r[1] = y[b]-y[a];
    float d = (float)Math.sqrt(r[0]*r[0]+r[1]*r[1]);
    r[0]/=d;
    r[1]/=d;
    return r;
  }
  
  private void makeUnitv(float[] v) {
     double d =  Math.sqrt(v[0]*v[0]+v[1]*v[1]);
     v[0]/=d;
     v[1]/=d;
  }
    
  
  public float[] nicePoints(int a, int b, int c, float r) {
    float[] v1=unitv(b,a);
    float[] v2=unitv(b,c);
    float[] v = new float[] {v1[0]+v2[0], v1[1]+v2[1]};
    makeUnitv(v);
    float[] mid = new float[] {r*v[0],r*v[1]};
    float[] perp = new float[] {mid[1]*2, -mid[0]*2};
    if ((perp[0]*v1[0]+perp[1]*v1[1])<0) {
       perp[0] = -perp[0];
       perp[1] = -perp[1];
    }
    float[] res = new float[] {mid[0]+perp[0]+x[b],mid[1]+perp[1]+y[b], mid[0]+x[b],mid[1]+y[b],mid[0]-perp[0]+x[b],mid[1]-perp[1]+y[b]};
    if (a==b || b==c) {
      res[0] = x[b];
      res[1] = y[b];
      res[2] = x[b];
      res[3] = y[b];
      res[4] = x[b];
      res[5] = y[b];       
    }
    return res;
  }
  
  private void applyTrans(float[] coords, float[] trans) {
     for (int i=0; i<coords.length; i+=2) {
        coords[i]=coords[i]*trans[0]+trans[1];
        coords[i+1]=coords[i+1]*trans[0]+trans[2];
     }
  }
  
  public void drawPathBezier(int[] path, float r) {
    float[] trans = getTrans();
    noFill();
    for (int i=0; i<path.length-1; i++) {
      float[] start = nicePoints(path[i>0 ? i-1 :0], path[i], path[i+1], r);
      float[] end = nicePoints(path[i], path[i+1],path[i<path.length-2 ? i+2 : i+1], r);
      applyTrans(start, trans);
      applyTrans(end, trans);
      bezier(start[2],start[3], start[4], start[5], end[0], end[1], end[2], end[3]);
      //line(start[2],start[3], start[4], start[5]);
      //line(start[4], start[5], end[0], end[1]);
      //line(end[0], end[1], end[2], end[3]);
    }
  }
}

Graph g;

void setup() {
  size(800,800);
  g = new Graph("hyperwuerfel.txt");
//  g = new Graph("wuerfel.txt");
//  g = new Graph("wuerfel-mit-diagonalen.txt");
//  g = new Graph("haus.txt");
//  g = new Graph("shortest.txt");
  g.draw();
  int[] path = g.eulerpath();
  if (path != null) {
    //println(path);
    stroke(255,0,0);
    g.drawPathBezier(path, 0.02);
  }
}

void loop() {
    
}